/*
 * Decompiled with CFR 0.152.
 */
package net.neoforged.neoform.runtime.cli;

import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import net.neoforged.neoform.runtime.utils.AnsiColor;
import net.neoforged.neoform.runtime.utils.HashingUtil;
import net.neoforged.neoform.runtime.utils.Logger;

public class LockManager {
    private static final Logger LOG = Logger.create();
    private final Path lockDirectory;
    private boolean verbose;

    public LockManager(Path lockDirectory) throws IOException {
        Files.createDirectories(lockDirectory, new FileAttribute[0]);
        this.lockDirectory = lockDirectory;
    }

    private Path getLockFile(String key) {
        return this.lockDirectory.resolve("_" + HashingUtil.sha1(key) + ".lock");
    }

    public Lock lock(String key) {
        FileLock fileLock;
        Path lockFile = this.getLockFile(key);
        FileChannel channel = null;
        int attempt = 0;
        while (channel == null) {
            try {
                ++attempt;
                channel = FileChannel.open(lockFile, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
            }
            catch (AccessDeniedException e) {
                if (attempt > 5) {
                    throw new RuntimeException("Failed to create lock-file " + String.valueOf(lockFile) + ": " + e.getMessage(), e);
                }
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ignored) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e);
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to create lock-file " + String.valueOf(lockFile) + ": " + e.getMessage(), e);
            }
        }
        Logger.IndeterminateSpinner spinner = null;
        while (true) {
            try {
                fileLock = channel.tryLock();
                if (fileLock != null) {
                    break;
                }
            }
            catch (OverlappingFileLockException overlappingFileLockException) {
            }
            catch (IOException e) {
                try {
                    channel.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw new RuntimeException(e);
            }
            if (spinner == null) {
                spinner = LOG.spinner("Waiting for lock on " + key);
            } else {
                spinner.tick();
            }
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                if (spinner != null) {
                    spinner.end();
                }
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        }
        if (spinner != null) {
            spinner.end();
        }
        if (this.verbose) {
            LOG.println(String.valueOf((Object)AnsiColor.MUTED) + " Acquired lock for " + key + String.valueOf((Object)AnsiColor.RESET));
        }
        return new Lock(fileLock);
    }

    public void performMaintenance() {
        FileTime newestToDelete = FileTime.from(Instant.now().minus(24L, ChronoUnit.HOURS));
        AtomicInteger lockFilesDeleted = new AtomicInteger();
        try (Stream<Path> stream = Files.list(this.lockDirectory);){
            stream.filter(f -> {
                String filename = f.getFileName().toString();
                return filename.startsWith("_") && filename.endsWith(".lock");
            }).filter(f -> {
                try {
                    BasicFileAttributes attributes = Files.readAttributes(f, BasicFileAttributes.class, new LinkOption[0]);
                    return attributes.isRegularFile() && attributes.lastModifiedTime().compareTo(newestToDelete) < 0;
                }
                catch (IOException ignored) {
                    return false;
                }
            }).forEach(f -> {
                try {
                    Files.delete(f);
                    lockFilesDeleted.incrementAndGet();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            });
        }
        catch (IOException iOException) {
            // empty catch block
        }
        if (lockFilesDeleted.get() > 0) {
            LOG.println(String.valueOf((Object)AnsiColor.MUTED) + " Deleted " + lockFilesDeleted.get() + " outdated lock files");
        }
    }

    public boolean isVerbose() {
        return this.verbose;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public static class Lock
    implements AutoCloseable {
        private final FileLock fileLock;

        public Lock(FileLock fileLock) {
            this.fileLock = fileLock;
        }

        @Override
        public void close() {
            try {
                this.fileLock.release();
            }
            catch (IOException ignored) {
                System.err.println("Failed to release lock on " + this.fileLock.channel().toString());
            }
            try {
                this.fileLock.channel().close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

