/*
 * Decompiled with CFR 0.152.
 */
package net.neoforged.neoform.runtime.downloads;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import net.neoforged.neoform.runtime.downloads.DownloadManager;
import net.neoforged.neoform.runtime.downloads.DownloadSpec;
import net.neoforged.neoform.runtime.downloads.DownloadsFailedException;
import net.neoforged.neoform.runtime.utils.FileUtil;
import net.neoforged.neoform.runtime.utils.Logger;
import net.neoforged.neoform.runtime.utils.StringUtil;
import org.jetbrains.annotations.Nullable;

public class ParallelDownloader
implements AutoCloseable {
    private static final Logger LOG = Logger.create();
    private static final ThreadFactory DOWNLOAD_THREAD_FACTORY = Thread.ofVirtual().name("parallel-download", 1L).factory();
    private final DownloadManager downloadManager;
    private final Semaphore semaphore;
    @Nullable
    private final ExecutorService executor;
    private final AtomicInteger downloadsDone = new AtomicInteger();
    private final AtomicInteger copiesDone = new AtomicInteger();
    private final AtomicLong bytesDownloaded = new AtomicLong();
    private final AtomicLong bytesCopied = new AtomicLong();
    private final List<Exception> errors = new ArrayList<Exception>();
    private final Path destination;
    private final int estimatedTotal;
    private volatile List<Path> localSources = List.of();

    public ParallelDownloader(DownloadManager downloadManager, int concurrentDownloads, Path destination, int estimatedTotal) {
        this.downloadManager = downloadManager;
        this.destination = destination;
        this.estimatedTotal = estimatedTotal;
        if (concurrentDownloads < 1) {
            throw new IllegalStateException("Cannot set concurrent downloads to less than 1: " + concurrentDownloads);
        }
        if (concurrentDownloads == 1) {
            this.executor = null;
            this.semaphore = null;
        } else {
            this.executor = Executors.newThreadPerTaskExecutor(DOWNLOAD_THREAD_FACTORY);
            this.semaphore = new Semaphore(concurrentDownloads);
        }
    }

    public void setLocalSources(List<Path> localSources) {
        if (localSources.contains(this.destination)) {
            localSources = new ArrayList<Path>(localSources);
            localSources.remove(this.destination);
        }
        this.localSources = localSources;
    }

    public void submitDownload(DownloadSpec spec, String relativeDestination) throws DownloadsFailedException {
        if (this.executor != null && this.semaphore != null) {
            this.executor.execute(() -> {
                boolean hasAcquired = false;
                try {
                    this.semaphore.acquire();
                    hasAcquired = true;
                    this.download(spec, relativeDestination);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                catch (Exception e) {
                    List<Exception> list = this.errors;
                    synchronized (list) {
                        this.errors.add(e);
                    }
                }
                finally {
                    if (hasAcquired) {
                        this.semaphore.release();
                    }
                }
            });
        } else {
            try {
                this.download(spec, relativeDestination);
            }
            catch (IOException e) {
                throw new DownloadsFailedException(List.of(e));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void download(DownloadSpec spec, String relativePath) throws IOException {
        Path objectDestination = this.destination.resolve(relativePath);
        try {
            for (Path localSource : this.localSources) {
                Path existingFile = localSource.resolve(relativePath);
                if (!Files.isRegularFile(existingFile, new LinkOption[0]) || Files.size(existingFile) != (long)spec.size()) continue;
                FileUtil.safeCopy(existingFile, objectDestination);
                this.bytesCopied.addAndGet(Files.size(objectDestination));
                this.copiesDone.incrementAndGet();
                return;
            }
            if (this.downloadManager.download(spec, objectDestination, true)) {
                this.bytesDownloaded.addAndGet(spec.size());
            }
        }
        finally {
            int finished = this.downloadsDone.incrementAndGet();
            if (finished % 100 == 0) {
                LOG.println(finished + "/" + this.estimatedTotal + " downloads");
            }
        }
    }

    @Override
    public void close() throws DownloadsFailedException {
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                while (!this.executor.awaitTermination(1L, TimeUnit.MINUTES)) {
                    Thread.yield();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
                throw new DownloadsFailedException(List.of(e));
            }
        }
        if (this.downloadsDone.get() > 0) {
            LOG.println("Downloaded " + this.downloadsDone.get() + " files with a total size of " + StringUtil.formatBytes(this.bytesDownloaded.get()));
        }
        if (this.copiesDone.get() > 0) {
            LOG.println("Copied " + this.copiesDone.get() + " files with a total size of " + StringUtil.formatBytes(this.bytesCopied.get()));
        }
        if (!this.errors.isEmpty()) {
            throw new DownloadsFailedException(this.errors);
        }
    }
}

