/*
 * Decompiled with CFR 0.152.
 */
package net.neoforged.neoform.runtime.cache;

import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.neoforged.neoform.runtime.cache.CacheKey;
import net.neoforged.neoform.runtime.graph.ExecutionNode;
import net.neoforged.neoform.runtime.graph.NodeOutput;
import net.neoforged.neoform.runtime.utils.AnsiColor;
import net.neoforged.neoform.runtime.utils.FileUtil;
import net.neoforged.neoform.runtime.utils.FilenameUtil;
import net.neoforged.neoform.runtime.utils.Logger;
import net.neoforged.neoform.runtime.utils.StringUtil;
import org.jetbrains.annotations.Nullable;

public class CacheManager
implements AutoCloseable {
    private static final Logger LOG = Logger.create();
    private static final DateTimeFormatter WORKSPACE_NAME_FORMAT = DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss");
    private final Path homeDir;
    private final Path artifactCacheDir;
    private final Path intermediateResultsDir;
    private final Path assetsDir;
    private final Path workspacesDir;
    private long maxAgeInHours = 744L;
    private long maxSize = 0x40000000L;
    private boolean disabled;
    private boolean analyzeMisses;
    private boolean verbose;

    public CacheManager(Path homeDir, @Nullable Path assetsDir, Path workspacesDir) throws IOException {
        this.homeDir = homeDir;
        Files.createDirectories(homeDir, new FileAttribute[0]);
        this.artifactCacheDir = homeDir.resolve("artifacts");
        this.intermediateResultsDir = homeDir.resolve("intermediate_results");
        this.assetsDir = Objects.requireNonNullElse(assetsDir, homeDir.resolve("assets"));
        this.workspacesDir = workspacesDir;
    }

    public void performMaintenance() throws IOException {
        if (!Files.exists(this.homeDir, new LinkOption[0])) {
            return;
        }
        Path cacheLock = this.homeDir.resolve("nfrt_cache_cleanup.state");
        try (FileChannel channel = FileChannel.open(cacheLock, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            FileLock lock;
            try {
                lock = channel.tryLock();
            }
            catch (OverlappingFileLockException ignored) {
                lock = null;
            }
            if (lock != null) {
                Duration interval;
                FileTime lastModified = Files.getLastModifiedTime(cacheLock, LinkOption.NOFOLLOW_LINKS);
                Duration age = Duration.between(lastModified.toInstant(), Instant.now());
                if (age.compareTo(interval = Duration.ofHours(24L)) < 0) {
                    if (this.verbose) {
                        LOG.println("Not performing routine maintenance since the last maintenance was " + String.valueOf((Object)AnsiColor.BOLD) + StringUtil.formatDuration(age) + " ago" + String.valueOf((Object)AnsiColor.RESET));
                    }
                    return;
                }
                LOG.println("Performing periodic cache maintenance on " + String.valueOf(this.homeDir));
                this.cleanUpIntermediateResults();
                Files.setLastModifiedTime(cacheLock, FileTime.from(Instant.now()));
                return;
            }
            LOG.println("Cache maintenance is already performed by another process.");
        }
    }

    public void cleanUpAll() throws IOException {
        this.cleanUpIntermediateResults();
    }

    public void cleanUpIntermediateResults() throws IOException {
        if (!Files.exists(this.intermediateResultsDir, new LinkOption[0])) {
            return;
        }
        LOG.println("Cleaning intermediate results cache in " + String.valueOf(this.intermediateResultsDir));
        LOG.println(" Maximum age: " + this.maxAgeInHours + "h");
        LOG.println(" Maximum cache size: " + StringUtil.formatBytes(this.maxSize));
        final ArrayList entries = new ArrayList(1000);
        final HashSet expiredEntryPrefixes = new HashSet();
        final Instant now = Instant.now();
        Files.walkFileTree(this.intermediateResultsDir, Set.of(), 1, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (attrs.isRegularFile()) {
                    String filename = file.getFileName().toString();
                    Matcher m = CacheKey.FILENAME_PREFIX_PATTERN.matcher(filename);
                    if (m.find()) {
                        long ageInHours;
                        String cacheKey = m.group(1);
                        record CacheEntry(Path file, String filename, String cacheKey, long lastModified, long size) {
                        }
                        entries.add(new CacheEntry(file, filename, cacheKey, attrs.lastModifiedTime().toMillis(), attrs.size()));
                        if (filename.substring(cacheKey.length()).equals(".txt") && (ageInHours = Duration.between(attrs.lastModifiedTime().toInstant(), now).toHours()) > CacheManager.this.maxAgeInHours) {
                            expiredEntryPrefixes.add(cacheKey);
                        }
                    } else {
                        LOG.println("  Unrecognized file in cache: " + String.valueOf(file));
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
        long totalSize = entries.stream().mapToLong(CacheEntry::size).sum();
        LOG.println(" " + String.valueOf((Object)AnsiColor.MUTED) + entries.size() + " files found" + String.valueOf((Object)AnsiColor.RESET));
        LOG.println(" " + String.valueOf((Object)AnsiColor.MUTED) + StringUtil.formatBytes(totalSize) + " overall size" + String.valueOf((Object)AnsiColor.RESET));
        LOG.println(" " + String.valueOf((Object)AnsiColor.MUTED) + expiredEntryPrefixes.size() + " expired keys found" + String.valueOf((Object)AnsiColor.RESET));
        if (!expiredEntryPrefixes.isEmpty()) {
            long freedSpace = 0L;
            long deletedEntries = 0L;
            Iterator it = entries.iterator();
            while (it.hasNext()) {
                CacheEntry item = (CacheEntry)it.next();
                if (!expiredEntryPrefixes.contains(item.cacheKey)) continue;
                if (this.verbose) {
                    LOG.println(" Deleting " + item.filename);
                }
                try {
                    Files.delete(item.file);
                }
                catch (IOException e) {
                    System.err.println("Failed to delete cache entry " + String.valueOf(item.file));
                    continue;
                }
                freedSpace += item.size;
                ++deletedEntries;
                it.remove();
            }
            LOG.println("Freed up " + String.valueOf((Object)AnsiColor.BOLD) + StringUtil.formatBytes(freedSpace) + String.valueOf((Object)AnsiColor.RESET) + " by deleting " + String.valueOf((Object)AnsiColor.BOLD) + deletedEntries + " expired entries" + String.valueOf((Object)AnsiColor.RESET));
            totalSize -= freedSpace;
        }
        if (totalSize <= this.maxSize) {
            return;
        }
        LOG.println("Cache size exceeds target size. Deleting oldest entries first.");
        ArrayList<List<CacheEntry>> groupedEntries = new ArrayList<List<CacheEntry>>(entries.stream().collect(Collectors.groupingBy(CacheEntry::cacheKey)).values());
        groupedEntries.sort(Comparator.comparingLong(group -> group.stream().mapToLong(CacheEntry::size).sum()).reversed());
        long freedSpace = 0L;
        int deletedEntries = 0;
        for (List<CacheEntry> group2 : groupedEntries) {
            if (totalSize <= this.maxSize) break;
            for (CacheEntry item : group2) {
                if (this.verbose) {
                    LOG.println(" Deleting " + item.filename);
                }
                try {
                    Files.delete(item.file);
                }
                catch (IOException e) {
                    System.err.println("Failed to delete cache entry " + String.valueOf(item.file));
                    continue;
                }
                freedSpace += item.size;
                totalSize -= item.size;
                ++deletedEntries;
            }
        }
        LOG.println("Freed up " + String.valueOf((Object)AnsiColor.BOLD) + StringUtil.formatBytes(freedSpace) + String.valueOf((Object)AnsiColor.RESET) + " by deleting " + String.valueOf((Object)AnsiColor.BOLD) + deletedEntries + " entries" + String.valueOf((Object)AnsiColor.RESET));
    }

    public boolean restoreOutputsFromCache(ExecutionNode node, CacheKey cacheKey, Map<String, Path> outputValues) throws IOException {
        if (this.disabled) {
            return false;
        }
        Path intermediateCacheDir = this.getIntermediateResultsDir();
        Path cacheMarkerFile = this.getCacheMarkerFile(cacheKey);
        Files.createDirectories(intermediateCacheDir, new FileAttribute[0]);
        if (Files.isRegularFile(cacheMarkerFile, new LinkOption[0])) {
            boolean complete = true;
            for (Map.Entry<String, NodeOutput> entry : node.outputs().entrySet()) {
                String filename = String.valueOf(cacheKey) + "_" + entry.getKey() + node.getRequiredOutput(entry.getKey()).type().getExtension();
                Path cachedFile = intermediateCacheDir.resolve(filename);
                if (Files.isRegularFile(cachedFile, new LinkOption[0])) {
                    outputValues.put(entry.getKey(), cachedFile);
                    continue;
                }
                System.err.println("Cache for " + node.id() + " is incomplete. Missing: " + filename);
                outputValues.clear();
                complete = false;
                break;
            }
            if (complete) {
                Files.setLastModifiedTime(cacheMarkerFile, FileTime.from(Instant.now()));
            }
            return complete;
        }
        if (this.analyzeMisses) {
            this.analyzeCacheMiss(cacheKey);
        }
        return false;
    }

    public void saveOutputs(ExecutionNode node, CacheKey cacheKey, HashMap<String, Path> outputValues) throws IOException {
        if (this.disabled) {
            return;
        }
        Path intermediateCacheDir = this.getIntermediateResultsDir();
        HashMap<String, Path> finalOutputValues = new HashMap<String, Path>(outputValues.size());
        for (Map.Entry<String, Path> entry : outputValues.entrySet()) {
            String filename = String.valueOf(cacheKey) + "_" + entry.getKey() + node.getRequiredOutput(entry.getKey()).type().getExtension();
            Path cachedPath = intermediateCacheDir.resolve(filename);
            FileUtil.atomicMove(entry.getValue(), cachedPath);
            finalOutputValues.put(entry.getKey(), cachedPath);
        }
        outputValues.putAll(finalOutputValues);
        cacheKey.write(this.getCacheMarkerFile(cacheKey));
    }

    private Path getCacheMarkerFile(CacheKey cacheKey) {
        return this.getIntermediateResultsDir().resolve(cacheKey.type() + "_" + cacheKey.hashValue() + ".txt");
    }

    private Path getIntermediateResultsDir() {
        return this.intermediateResultsDir;
    }

    public Path getArtifactCacheDir() {
        return this.artifactCacheDir;
    }

    public Path getAssetsDir() {
        return this.assetsDir;
    }

    private void analyzeCacheMiss(CacheKey cacheKey) {
        Path intermediateCacheDir = this.getIntermediateResultsDir();
        ArrayList<CacheEntry> cacheEntries = new ArrayList<CacheEntry>(CacheManager.getCacheEntries(intermediateCacheDir, cacheKey.type()));
        LOG.println("  " + cacheEntries.size() + " existing cache entries for " + cacheKey.type());
        IdentityHashMap<CacheEntry, List<CacheKey.Delta>> deltasByCacheEntry = new IdentityHashMap<CacheEntry, List<CacheKey.Delta>>(cacheEntries.size());
        for (CacheEntry cacheEntry : cacheEntries) {
            deltasByCacheEntry.put(cacheEntry, cacheKey.getDiff(cacheEntry.cacheKey()));
        }
        cacheEntries.sort(Comparator.comparingInt(value -> ((List)deltasByCacheEntry.get(value)).size()));
        for (CacheEntry cacheEntry : cacheEntries) {
            int diffCount = ((List)deltasByCacheEntry.get(cacheEntry)).size();
            LOG.println("    " + cacheEntry.filename + " " + String.valueOf(cacheEntry.lastModified) + " " + diffCount + " deltas");
        }
        if (!cacheEntries.isEmpty()) {
            LOG.println("  Detailed delta for cache entry with best match:");
            for (CacheKey.Delta delta : (List)deltasByCacheEntry.get(cacheEntries.getFirst())) {
                LOG.println("    " + String.valueOf((Object)AnsiColor.UNDERLINE) + delta.key() + String.valueOf((Object)AnsiColor.RESET));
                LOG.println(String.valueOf((Object)AnsiColor.MUTED) + "      New: " + String.valueOf((Object)AnsiColor.RESET) + CacheManager.print(delta.ours()));
                LOG.println(String.valueOf((Object)AnsiColor.MUTED) + "      Old: " + String.valueOf((Object)AnsiColor.RESET) + CacheManager.print(delta.theirs()));
            }
        }
    }

    private static String print(@Nullable CacheKey.AnnotatedValue value) {
        if (value == null) {
            return String.valueOf((Object)AnsiColor.MUTED) + "absent" + String.valueOf((Object)AnsiColor.RESET);
        }
        if (value.annotation() != null) {
            return value.value() + String.valueOf((Object)AnsiColor.MUTED) + " (" + value.annotation() + ")" + String.valueOf((Object)AnsiColor.RESET);
        }
        return value.value();
    }

    private static List<CacheEntry> getCacheEntries(Path intermediateCacheDir, String type) {
        List<CacheEntry> list;
        block8: {
            Pattern filenamePattern = Pattern.compile(Pattern.quote(type) + "_[0-9a-f]+\\.txt");
            Stream<Path> stream = Files.list(intermediateCacheDir);
            try {
                list = stream.filter(f -> filenamePattern.matcher(f.getFileName().toString()).matches()).map(p -> {
                    try {
                        return new CacheEntry(p.getFileName().toString(), Files.getLastModifiedTime(p, new LinkOption[0]), CacheKey.read(p));
                    }
                    catch (Exception e) {
                        System.err.println("  Failed to read cache-key " + String.valueOf(p) + " for analysis");
                        return null;
                    }
                }).filter(Objects::nonNull).toList();
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ignored) {
                    return List.of();
                }
            }
            stream.close();
        }
        return list;
    }

    public Path createWorkspace(String stepName) throws IOException {
        Files.createDirectories(this.workspacesDir, new FileAttribute[0]);
        String timestamp = WORKSPACE_NAME_FORMAT.format(LocalDateTime.now());
        String workspaceBasename = timestamp + "_" + FilenameUtil.sanitizeForFilename(stepName);
        Path workspaceDir = this.workspacesDir.resolve(workspaceBasename);
        for (int i = 1; i <= 999; ++i) {
            try {
                Files.createDirectory(workspaceDir, new FileAttribute[0]);
                break;
            }
            catch (FileAlreadyExistsException e) {
                workspaceDir = workspaceDir.resolveSibling(workspaceBasename + "_" + String.format(Locale.ROOT, "%03d", i));
                continue;
            }
        }
        if (!Files.isDirectory(workspaceDir, new LinkOption[0])) {
            throw new IOException("Failed to create a suitable workspace directory " + String.valueOf(workspaceDir));
        }
        return workspaceDir;
    }

    public boolean isDisabled() {
        return this.disabled;
    }

    public void setDisabled(boolean disabled) {
        this.disabled = disabled;
    }

    public boolean isAnalyzeMisses() {
        return this.analyzeMisses;
    }

    public void setAnalyzeMisses(boolean analyzeMisses) {
        this.analyzeMisses = analyzeMisses;
    }

    public boolean isVerbose() {
        return this.verbose;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public void close() throws Exception {
    }

    record CacheEntry(String filename, FileTime lastModified, CacheKey cacheKey) {
    }
}

