/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.neoforged.neoforge.client.event;

import com.mojang.blaze3d.vertex.PoseStack;
import net.minecraft.client.renderer.SubmitNodeCollector;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.item.ItemStack;
import net.neoforged.bus.api.Event;
import net.neoforged.bus.api.ICancellableEvent;
import net.neoforged.fml.LogicalSide;
import net.neoforged.neoforge.common.NeoForge;
import org.jetbrains.annotations.ApiStatus;

/**
 * Fired before a hand is rendered in the first person view.
 *
 * <p>This event is {@linkplain ICancellableEvent cancellable}, and does not {@linkplain HasResult have a result}.
 * If this event is cancelled, then the hand will not be rendered.</p>
 *
 * <p>This event is fired on the {@linkplain NeoForge#EVENT_BUS main Forge event bus},
 * only on the {@linkplain LogicalSide#CLIENT logical client}.</p>
 *
 * @see RenderArmEvent
 */
public class RenderHandEvent extends Event implements ICancellableEvent {
    private final InteractionHand hand;
    private final PoseStack poseStack;
    private final SubmitNodeCollector submitNodeCollector;
    private final int packedLight;
    private final float partialTick;
    private final float interpolatedPitch;
    private final float swingProgress;
    private final float equipProgress;
    private final ItemStack stack;

    @ApiStatus.Internal
    public RenderHandEvent(InteractionHand hand, PoseStack poseStack, SubmitNodeCollector submitNodeCollector, int packedLight,
            float partialTick, float interpolatedPitch,
            float swingProgress, float equipProgress, ItemStack stack) {
        this.hand = hand;
        this.poseStack = poseStack;
        this.submitNodeCollector = submitNodeCollector;
        this.packedLight = packedLight;
        this.partialTick = partialTick;
        this.interpolatedPitch = interpolatedPitch;
        this.swingProgress = swingProgress;
        this.equipProgress = equipProgress;
        this.stack = stack;
    }

    /**
     * {@return the hand being rendered}
     */
    public InteractionHand getHand() {
        return hand;
    }

    /**
     * {@return the pose stack used for rendering}
     */
    public PoseStack getPoseStack() {
        return poseStack;
    }

    /**
     * {@return the submit node collector}
     */
    public SubmitNodeCollector getSubmitNodeCollector() {
        return submitNodeCollector;
    }

    /**
     * {@return the amount of packed (sky and block) light for rendering}
     *
     * @see net.minecraft.util.LightCoordsUtil
     */
    public int getPackedLight() {
        return packedLight;
    }

    /**
     * {@return the partial tick}
     */
    public float getPartialTick() {
        return partialTick;
    }

    /**
     * {@return the interpolated pitch of the player entity}
     */
    public float getInterpolatedPitch() {
        return interpolatedPitch;
    }

    /**
     * {@return the swing progress of the hand being rendered}
     */
    public float getSwingProgress() {
        return swingProgress;
    }

    /**
     * {@return the progress of the equip animation, from {@code 0.0} to {@code 1.0}}
     */
    public float getEquipProgress() {
        return equipProgress;
    }

    /**
     * {@return the item stack to be rendered}
     */
    public ItemStack getItemStack() {
        return stack;
    }
}
