/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.neoforged.neoforge.client.extensions.common;

import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.client.particle.ParticleEngine;
import net.minecraft.client.sounds.SoundManager;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.tags.FluidTags;
import net.minecraft.world.effect.MobEffects;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.ai.attributes.AttributeInstance;
import net.minecraft.world.entity.ai.attributes.Attributes;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelReader;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.material.FluidState;
import net.minecraft.world.phys.HitResult;
import net.neoforged.fml.LogicalSide;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector3d;

/**
 * {@linkplain LogicalSide#CLIENT Client-only} extensions to {@link Block}.
 *
 * @see net.neoforged.neoforge.client.extensions.common.RegisterClientExtensionsEvent
 */
public interface IClientBlockExtensions {
    IClientBlockExtensions DEFAULT = new IClientBlockExtensions() {};

    static IClientBlockExtensions of(BlockState state) {
        return of(state.getBlock());
    }

    static IClientBlockExtensions of(Block block) {
        return ClientExtensionsManager.BLOCK_EXTENSIONS.getOrDefault(block, DEFAULT);
    }

    /**
     * Spawn a digging particle effect in the level, this is a wrapper
     * around EffectRenderer.addBlockHitEffects to allow the block more
     * control over the particles. Useful when you have entirely different
     * texture sheets for different sides/locations in the level.
     *
     * @param state   The current state
     * @param level   The current level
     * @param target  The target the player is looking at {x/y/z/side/sub}, if available
     * @param manager A reference to the current particle manager.
     * @return True to prevent vanilla digging particles form spawning.
     */
    default boolean addHitEffects(BlockState state, Level level, @Nullable HitResult target, ParticleEngine manager) {
        return !state.shouldSpawnTerrainParticles();
    }

    /**
     * Spawn particles for when the block is destroyed. Due to the nature
     * of how this is invoked, the x/y/z locations are not always guaranteed
     * to host your block. So be sure to do proper sanity checks before assuming
     * that the location is this block.
     *
     * @param Level   The current Level
     * @param pos     Position to spawn the particle
     * @param manager A reference to the current particle manager.
     * @return True to prevent vanilla break particles from spawning.
     */
    default boolean addDestroyEffects(BlockState state, Level Level, BlockPos pos, ParticleEngine manager) {
        return !state.shouldSpawnTerrainParticles();
    }

    /**
     * Play hit sound(s) when the block is punched.
     *
     * @param state        The current state
     * @param level        The current level
     * @param pos          The position of the block
     * @param face         The face of the block that was hit
     * @param soundManager The sound manager to play the sound(s) through
     * @return True to prevent vanilla hit sounds from playing
     */
    default boolean playHitSound(BlockState state, Level level, BlockPos pos, Direction face, SoundManager soundManager) {
        return false;
    }

    /**
     * Play breaking sound(s) when the block is destroyed. This allows playing sounds dependent on BE data
     * as it is called before the block and BE are actually removed on the client.
     *
     * @param state The current state
     * @param level The current level
     * @param pos   The position of the block
     * @return True to prevent vanilla break sounds from playing
     */
    default boolean playBreakSound(BlockState state, Level level, BlockPos pos) {
        return false;
    }

    /**
     * NOT CURRENTLY IMPLEMENTED
     * <p>
     * Use this to change the fog color used when the entity is "inside" a material.
     * Vec3d is used here as "r/g/b" 0 - 1 values.
     *
     * @param level         The level.
     * @param pos           The position at the entity viewport.
     * @param state         The state at the entity viewport.
     * @param entity        the entity
     * @param originalColor The current fog color, You are not expected to use this, Return as the default if applicable.
     * @return The new fog color.
     */
    default Vector3d getFogColor(BlockState state, LevelReader level, BlockPos pos, Entity entity, Vector3d originalColor, float partialTick) {
        FluidState fluidState = level.getFluidState(pos);
        if (fluidState.is(FluidTags.WATER)) {
            float f12 = 0.0F;

            if (entity instanceof LivingEntity) {
                LivingEntity ent = (LivingEntity) entity;
                AttributeInstance attributeinstance = ent.getAttribute(Attributes.OXYGEN_BONUS);
                f12 = (float) (attributeinstance != null ? attributeinstance.getValue() : 0) * 0.2F;

                if (ent.hasEffect(MobEffects.WATER_BREATHING)) {
                    f12 = f12 * 0.3F + 0.6F;
                }
            }
            return new Vector3d(0.02F + f12, 0.02F + f12, 0.2F + f12);
        } else if (fluidState.is(FluidTags.LAVA)) {
            return new Vector3d(0.6F, 0.1F, 0.0F);
        }
        return originalColor;
    }

    /**
     * Returns true if the breaking particles created from the {@link BlockState} passed should be tinted with biome colors.
     * 
     * @param state The state of this block
     * @param level The level the particles are spawning in
     * @param pos   The position of the block
     * @return {@code true} if the particles should be tinted.
     */
    default boolean areBreakingParticlesTinted(BlockState state, ClientLevel level, BlockPos pos) {
        return !state.is(Blocks.GRASS_BLOCK);
    }
}
