/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.neoforged.neoforge.network.filters;

import com.google.common.collect.ImmutableMap;
import io.netty.channel.ChannelHandler;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import net.minecraft.network.Connection;
import net.minecraft.network.ConnectionProtocol;
import net.minecraft.network.protocol.Packet;
import net.minecraft.network.protocol.PacketFlow;
import net.minecraft.network.protocol.common.ClientboundUpdateTagsPacket;
import net.minecraft.network.protocol.game.ClientboundLoginPacket;
import net.minecraft.network.protocol.game.ClientboundUpdateAdvancementsPacket;
import net.minecraft.network.protocol.game.ClientboundUpdateRecipesPacket;
import org.jetbrains.annotations.Nullable;

/**
 * Network filter for forge-forge connections.
 */
@ChannelHandler.Sharable
public class NeoForgeConnectionNetworkFilter extends VanillaPacketFilter {
    public NeoForgeConnectionNetworkFilter(@Nullable Connection manager) {
        super(buildHandlers(manager));
    }

    private static Map<Class<? extends Packet<?>>, BiConsumer<Packet<?>, List<? super Packet<?>>>> buildHandlers(@Nullable Connection manager) {

        VanillaPacketSplitter.RemoteCompatibility compatibility = manager == null ? VanillaPacketSplitter.RemoteCompatibility.ABSENT : VanillaPacketSplitter.getRemoteCompatibility(manager);
        if (compatibility == VanillaPacketSplitter.RemoteCompatibility.ABSENT) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder<Class<? extends Packet<?>>, BiConsumer<Packet<?>, List<? super Packet<?>>>> builder = ImmutableMap.<Class<? extends Packet<?>>, BiConsumer<Packet<?>, List<? super Packet<?>>>>builder()
                .put(ClientboundUpdateRecipesPacket.class, NeoForgeConnectionNetworkFilter::splitPacket)
                .put(ClientboundUpdateTagsPacket.class, NeoForgeConnectionNetworkFilter::splitPacket)
                .put(ClientboundUpdateAdvancementsPacket.class, NeoForgeConnectionNetworkFilter::splitPacket)
                .put(ClientboundLoginPacket.class, NeoForgeConnectionNetworkFilter::splitPacket); // When there are many dynamic registry entries that packet is BIG

        return builder.build();
    }

    @Override
    protected boolean isNecessary(Connection manager) {
        // not needed on local connections, because packets are not encoded to bytes there
        return !manager.isMemoryConnection() && VanillaPacketSplitter.isRemoteCompatible(manager);
    }

    private static void splitPacket(Packet<?> packet, List<? super Packet<?>> out) {
        VanillaPacketSplitter.appendPackets(
                ConnectionProtocol.PLAY, PacketFlow.CLIENTBOUND, packet, out);
    }

}
