/*
 * Decompiled with CFR 0.152.
 */
package net.neoforged.moddevgradle.internal.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.module.ModuleDescriptor;
import java.nio.charset.Charset;
import java.nio.file.AccessDeniedException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.HexFormat;
import java.util.List;
import java.util.Optional;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import org.gradle.api.GradleException;
import org.jetbrains.annotations.ApiStatus;

@ApiStatus.Internal
public final class FileUtils {
    private static final int MAX_TRIES = 2;

    private FileUtils() {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<String> getExplicitJavaModuleName(File file) throws IOException {
        try (JarFile jf = new JarFile(file, false, 1, JarFile.runtimeVersion());){
            JarEntry moduleInfoEntry = jf.getJarEntry("module-info.class");
            if (moduleInfoEntry != null) {
                try (InputStream in = jf.getInputStream(moduleInfoEntry);){
                    Optional<String> optional = Optional.of(ModuleDescriptor.read(in).name());
                    return optional;
                }
            }
            Manifest manifest = jf.getManifest();
            if (manifest == null) {
                Optional<String> optional = Optional.empty();
                return optional;
            }
            String automaticModuleName = manifest.getMainAttributes().getValue("Automatic-Module-Name");
            if (automaticModuleName == null) {
                Optional<String> optional = Optional.empty();
                return optional;
            }
            Optional<String> optional = Optional.of(automaticModuleName);
            return optional;
        }
        catch (Exception e) {
            throw new IOException("Failed to determine the Java module name of " + String.valueOf(file) + ": " + String.valueOf(e), e);
        }
    }

    public static String hashFile(File file, String algorithm) {
        try {
            MessageDigest digest = MessageDigest.getInstance(algorithm);
            try (DigestInputStream input = new DigestInputStream(new FileInputStream(file), digest);){
                input.transferTo(OutputStream.nullOutputStream());
            }
            return HexFormat.of().formatHex(digest.digest());
        }
        catch (Exception e) {
            throw new GradleException("Failed to hash file " + String.valueOf(file), (Throwable)e);
        }
    }

    public static void writeStringSafe(Path destination, String content, Charset charset) throws IOException {
        if (!charset.newEncoder().canEncode(content)) {
            throw new IllegalArgumentException("The given character set " + String.valueOf(charset) + " cannot represent this string: " + content);
        }
        try (OutputStream out = FileUtils.newSafeFileOutputStream(destination);){
            byte[] encodedContent = content.getBytes(charset);
            out.write(encodedContent);
        }
    }

    public static void writeLinesSafe(Path destination, List<String> lines, Charset charset) throws IOException {
        FileUtils.writeStringSafe(destination, String.join((CharSequence)"\n", lines), charset);
    }

    public static OutputStream newSafeFileOutputStream(final Path destination) throws IOException {
        String uniqueId = ProcessHandle.current().pid() + "." + Thread.currentThread().getId();
        final Path tempFile = destination.resolveSibling(destination.getFileName().toString() + "." + uniqueId + ".tmp");
        final boolean[] closed = new boolean[1];
        return new FilterOutputStream(Files.newOutputStream(tempFile, new OpenOption[0])){

            @Override
            public void close() throws IOException {
                try {
                    super.close();
                    if (!closed[0]) {
                        FileUtils.atomicMoveIfPossible(tempFile, destination);
                    }
                }
                finally {
                    try {
                        Files.deleteIfExists(tempFile);
                    }
                    catch (IOException iOException) {}
                    closed[0] = true;
                }
            }
        };
    }

    public static void atomicMove(Path source, Path destination) throws IOException {
        if (Files.isDirectory(destination, new LinkOption[0])) {
            throw new IOException("Cannot overwrite directory " + String.valueOf(destination));
        }
        try {
            FileUtils.atomicMoveIfPossible(source, destination);
        }
        catch (AccessDeniedException ex) {
            int tries = 0;
            while (true) {
                try {
                    Thread.sleep(10L * (long)tries);
                    FileUtils.atomicMoveIfPossible(source, destination);
                    return;
                }
                catch (AccessDeniedException ex2) {
                    if (tries == 1) {
                        throw ex;
                    }
                }
                catch (InterruptedException exInterrupt) {
                    Thread.currentThread().interrupt();
                    throw ex;
                }
                ++tries;
            }
        }
    }

    private static void atomicMoveIfPossible(Path source, Path destination) throws IOException {
        try {
            Files.move(source, destination, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AtomicMoveNotSupportedException ex) {
            Files.move(source, destination, StandardCopyOption.REPLACE_EXISTING);
        }
    }
}

