/*
 * Copyright (c) NeoForged and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.minecraftforge.common.damagesource;

import org.jetbrains.annotations.Nullable;

import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.ComponentUtils;
import net.minecraft.world.damagesource.CombatEntry;
import net.minecraft.world.damagesource.CombatTracker;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.damagesource.DeathMessageType;
import net.minecraft.world.entity.LivingEntity;

/**
 * An {@link IDeathMessageProvider} is used by custom {@link DeathMessageType} instances.<br>
 * This allows providing custom death messages based on the available parameters, instead of the hard-coded vanilla defaults.
 */
public interface IDeathMessageProvider
{
    /**
     * Default death message provider used by the vanilla {@link DeathMessageType}s.
     * 
     * @implNote Based off of the implementation in {@link CombatTracker#getDeathMessage()}.
     */
    IDeathMessageProvider DEFAULT = (entity, lastEntry, sigFall) -> {
        DamageSource dmgSrc = lastEntry.f_19250_();
        DeathMessageType msgType = dmgSrc.m_269415_().f_268472_();
        if (msgType == DeathMessageType.FALL_VARIANTS && sigFall != null)
        {
            return entity.m_21231_().m_289215_(sigFall, dmgSrc.m_7639_());
        }
        else if (msgType == DeathMessageType.INTENTIONAL_GAME_DESIGN)
        {
            String s = "death.attack." + dmgSrc.m_19385_();
            Component component = ComponentUtils.m_130748_(Component.m_237115_(s + ".link")).m_130948_(CombatTracker.f_268553_);
            return Component.m_237110_(s + ".message", entity.m_5446_(), component);
        }
        else // DeathMessageType.DEFAULT or DeathMessageType.FALL_VARIANTS and no fall was available.
        {
            return dmgSrc.m_6157_(entity);
        }
    };

    /**
     * Computes the death message from the available context.<br>
     * This method is not invoked if there are no available combat entries, since the damage source would not be available.
     * 
     * @param entity              The entity being killed.
     * @param lastEntry           The last entry from the entity's {@link CombatTracker}
     * @param mostSignificantFall The most significant fall inflicted to the entity, from {@link CombatTracker#getMostSignificantFall()}.
     * @return The death message for the slain entity.
     * @see {@link LivingEntity#getCombatTracker()} for additional context.
     */
    Component getDeathMessage(LivingEntity entity, CombatEntry lastEntry, @Nullable CombatEntry mostSignificantFall);
}
