/*
 * Decompiled with CFR 0.152.
 */
package net.neoforged.fml.loading.moddiscovery;

import com.mojang.logging.LogUtils;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.zip.ZipException;
import net.neoforged.fml.ModLoadingException;
import net.neoforged.fml.ModLoadingIssue;
import net.neoforged.fml.i18n.FMLTranslations;
import net.neoforged.fml.jarcontents.JarContents;
import net.neoforged.fml.loading.ImmediateWindowHandler;
import net.neoforged.fml.loading.LogMarkers;
import net.neoforged.fml.loading.UniqueModListBuilder;
import net.neoforged.fml.loading.moddiscovery.IncompatibleModReason;
import net.neoforged.fml.loading.moddiscovery.ModFile;
import net.neoforged.fml.util.ServiceLoaderUtil;
import net.neoforged.neoforgespi.ILaunchContext;
import net.neoforged.neoforgespi.language.IModInfo;
import net.neoforged.neoforgespi.locating.IDependencyLocator;
import net.neoforged.neoforgespi.locating.IDiscoveryPipeline;
import net.neoforged.neoforgespi.locating.IModFile;
import net.neoforged.neoforgespi.locating.IModFileCandidateLocator;
import net.neoforged.neoforgespi.locating.IModFileReader;
import net.neoforged.neoforgespi.locating.IncompatibleFileReporting;
import net.neoforged.neoforgespi.locating.ModFileDiscoveryAttributes;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;

public class ModDiscoverer {
    private static final Logger LOGGER = LogUtils.getLogger();
    private final List<IModFileCandidateLocator> modFileLocators;
    private final List<IDependencyLocator> dependencyLocators;
    private final List<IModFileReader> modFileReaders;
    private final ILaunchContext launchContext;

    public ModDiscoverer(ILaunchContext launchContext) {
        this(launchContext, List.of());
    }

    public ModDiscoverer(ILaunchContext launchContext, Collection<IModFileCandidateLocator> additionalModFileLocators) {
        this.launchContext = launchContext;
        this.modFileLocators = ServiceLoaderUtil.loadServices(launchContext, IModFileCandidateLocator.class, additionalModFileLocators);
        this.modFileReaders = ServiceLoaderUtil.loadServices(launchContext, IModFileReader.class);
        this.dependencyLocators = ServiceLoaderUtil.loadServices(launchContext, IDependencyLocator.class);
    }

    public Result discoverMods() {
        DiscoveryPipeline pipeline;
        LOGGER.debug(LogMarkers.SCAN, "Scanning for mods and other resources to load. We know {} ways to find mods", (Object)this.modFileLocators.size());
        List<ModFile> loadedFiles = new ArrayList<ModFile>();
        ArrayList<ModLoadingIssue> discoveryIssues = new ArrayList<ModLoadingIssue>();
        boolean successfullyLoadedMods = true;
        ImmediateWindowHandler.updateProgress("Discovering mod files");
        for (IModFileCandidateLocator locator : this.modFileLocators) {
            LOGGER.debug(LogMarkers.SCAN, "Trying locator {}", (Object)locator);
            ModFileDiscoveryAttributes defaultAttributes = ModFileDiscoveryAttributes.DEFAULT.withLocator(locator);
            pipeline = new DiscoveryPipeline(defaultAttributes, loadedFiles, discoveryIssues);
            try {
                locator.findCandidates(this.launchContext, pipeline);
            }
            catch (ModLoadingException e) {
                discoveryIssues.addAll(e.getIssues());
            }
            catch (Exception e) {
                discoveryIssues.add(ModLoadingIssue.error("fml.modloadingissue.technical_error", locator.toString() + " failed").withCause(e));
            }
            LOGGER.debug(LogMarkers.SCAN, "Locator {} found {} mods, {} warnings, {} errors and skipped {} candidates", new Object[]{locator, pipeline.successCount, pipeline.warningCount, pipeline.errorCount, pipeline.skipCount});
        }
        Map<Object, Object> modFilesMap = Collections.emptyMap();
        try {
            UniqueModListBuilder modsUniqueListBuilder = new UniqueModListBuilder(loadedFiles);
            UniqueModListBuilder.UniqueModListData uniqueModsData = modsUniqueListBuilder.buildUniqueList();
            uniqueModsData.discardedFiles().forEach(ModFile::close);
            modFilesMap = uniqueModsData.modFiles().stream().collect(Collectors.groupingBy(IModFile::getType));
            loadedFiles = uniqueModsData.modFiles();
        }
        catch (ModLoadingException exception) {
            LOGGER.error(LogMarkers.SCAN, "Failed to build unique mod list after mod discovery.", (Throwable)exception);
            discoveryIssues.addAll(exception.getIssues());
            successfullyLoadedMods = false;
            loadedFiles.forEach(ModFile::close);
        }
        if (successfullyLoadedMods) {
            LOGGER.debug(LogMarkers.SCAN, "Successfully Loaded {} mods. Attempting to load dependencies...", (Object)loadedFiles.size());
            for (IDependencyLocator locator : this.dependencyLocators) {
                try {
                    LOGGER.debug(LogMarkers.SCAN, "Trying locator {}", (Object)locator);
                    pipeline = new DiscoveryPipeline(ModFileDiscoveryAttributes.DEFAULT.withDependencyLocator(locator), loadedFiles, discoveryIssues);
                    locator.scanMods(List.copyOf(loadedFiles), pipeline);
                }
                catch (ModLoadingException exception) {
                    LOGGER.error(LogMarkers.SCAN, "Failed to load dependencies with locator {}", (Object)locator, (Object)exception);
                    discoveryIssues.addAll(exception.getIssues());
                }
            }
            try {
                UniqueModListBuilder modsAndDependenciesUniqueListBuilder = new UniqueModListBuilder(loadedFiles);
                UniqueModListBuilder.UniqueModListData uniqueModsAndDependenciesData = modsAndDependenciesUniqueListBuilder.buildUniqueList();
                uniqueModsAndDependenciesData.discardedFiles().forEach(ModFile::close);
                modFilesMap = uniqueModsAndDependenciesData.modFiles().stream().collect(Collectors.groupingBy(IModFile::getType));
            }
            catch (ModLoadingException exception) {
                LOGGER.error(LogMarkers.SCAN, "Failed to build unique mod list after dependency discovery.", (Throwable)exception);
                discoveryIssues.addAll(exception.getIssues());
                modFilesMap = loadedFiles.stream().collect(Collectors.groupingBy(IModFile::getType));
            }
        } else {
            LOGGER.error(LogMarkers.SCAN, "Mod Discovery failed. Skipping dependency discovery.");
        }
        LOGGER.info("\n     Mod List:\n\t\tName Version (Mod Id)\n\n{}", (Object)this.logReport(modFilesMap.values()));
        ArrayList<ModFile> modFiles = new ArrayList<ModFile>();
        modFilesMap.values().forEach(modFiles::addAll);
        return new Result(modFiles, discoveryIssues);
    }

    private String logReport(Collection<List<ModFile>> modFiles) {
        return modFiles.stream().flatMap(Collection::stream).filter(modFile -> !modFile.getModInfos().isEmpty()).sorted(Comparator.comparing(modFile -> modFile.getModInfos().getFirst().getDisplayName(), String.CASE_INSENSITIVE_ORDER)).map(this::fileToLine).collect(Collectors.joining("\n\t\t", "\t\t", ""));
    }

    private String fileToLine(IModFile mf) {
        IModInfo mainMod = mf.getModInfos().getFirst();
        return String.format(Locale.ENGLISH, "%s %s (%s)", mainMod.getDisplayName(), mainMod.getVersion(), mainMod.getModId());
    }

    private static void closeJarContents(JarContents jarContents) {
        try {
            jarContents.close();
        }
        catch (IOException e) {
            LOGGER.error("Failed to close jar contents {}", (Object)jarContents, (Object)e);
        }
    }

    private static boolean causeChainContains(Throwable e, Class<?> exceptionClass) {
        while (e != null) {
            if (exceptionClass.isInstance(e)) {
                return true;
            }
            e = e.getCause();
        }
        return false;
    }

    private class DiscoveryPipeline
    implements IDiscoveryPipeline {
        private final ModFileDiscoveryAttributes defaultAttributes;
        private final List<ModFile> loadedFiles;
        private final List<ModLoadingIssue> issues;
        private int successCount;
        private int errorCount;
        private int warningCount;
        private int skipCount;

        public DiscoveryPipeline(ModFileDiscoveryAttributes defaultAttributes, List<ModFile> loadedFiles, List<ModLoadingIssue> issues) {
            this.defaultAttributes = defaultAttributes;
            this.loadedFiles = loadedFiles;
            this.issues = issues;
        }

        @Override
        public Optional<IModFile> addPath(List<Path> groupedPaths, ModFileDiscoveryAttributes attributes, IncompatibleFileReporting reporting) {
            JarContents jarContents;
            Path primaryPath = groupedPaths.getFirst();
            if (!ModDiscoverer.this.launchContext.addLocated(primaryPath)) {
                LOGGER.debug("Skipping {} because it was already located earlier", (Object)primaryPath);
                ++this.skipCount;
                return Optional.empty();
            }
            try {
                jarContents = JarContents.ofPaths(groupedPaths);
            }
            catch (Exception e) {
                if (ModDiscoverer.causeChainContains(e, ZipException.class)) {
                    this.addIssue(ModLoadingIssue.error("fml.modloadingissue.brokenfile.invalidzip", new Object[0]).withAffectedPath(primaryPath).withCause(e));
                } else {
                    this.addIssue(ModLoadingIssue.error("fml.modloadingissue.brokenfile", new Object[0]).withAffectedPath(primaryPath).withCause(e));
                }
                return Optional.empty();
            }
            return this.addJarContent(jarContents, attributes, reporting);
        }

        @Override
        @Nullable
        public IModFile readModFile(JarContents jarContents, ModFileDiscoveryAttributes attributes) {
            for (IModFileReader reader : ModDiscoverer.this.modFileReaders) {
                IModFile provided = reader.read(jarContents, attributes);
                if (provided == null) continue;
                return provided;
            }
            throw new RuntimeException("No mod reader felt responsible for " + String.valueOf(jarContents.getPrimaryPath()));
        }

        @Override
        public Optional<IModFile> addJarContent(JarContents jarContents, ModFileDiscoveryAttributes attributes, IncompatibleFileReporting reporting) {
            attributes = this.defaultAttributes.merge(attributes);
            ArrayList<ModLoadingIssue> incompatibilityIssues = new ArrayList<ModLoadingIssue>();
            for (IModFileReader reader : ModDiscoverer.this.modFileReaders) {
                try {
                    IModFile provided = reader.read(jarContents, attributes);
                    if (provided == null) continue;
                    if (this.addModFile(provided)) {
                        return Optional.of(provided);
                    }
                    if (provided instanceof ModFile) {
                        ModFile modFile = (ModFile)provided;
                        modFile.close();
                    } else {
                        ModDiscoverer.closeJarContents(jarContents);
                    }
                    return Optional.empty();
                }
                catch (ModLoadingException e) {
                    incompatibilityIssues.addAll(e.getIssues());
                }
                catch (Exception e) {
                    ModDiscoverer.closeJarContents(jarContents);
                    this.addIssue(ModLoadingIssue.error("fml.modloadingissue.brokenfile", new Object[0]).withAffectedPath(jarContents.getPrimaryPath()).withCause(e));
                    return Optional.empty();
                }
            }
            if (reporting != IncompatibleFileReporting.IGNORE) {
                Optional<IncompatibleModReason> reason = IncompatibleModReason.detect(jarContents);
                if (reason.isPresent()) {
                    incompatibilityIssues.add(ModLoadingIssue.error(reason.get().getReason(), new Object[0]).withAffectedPath(jarContents.getPrimaryPath()));
                } else if (reporting != IncompatibleFileReporting.WARN_ON_KNOWN_INCOMPATIBILITY && incompatibilityIssues.isEmpty()) {
                    incompatibilityIssues.add(ModLoadingIssue.error("fml.modloadingissue.brokenfile.unknown", new Object[0]).withAffectedPath(jarContents.getPrimaryPath()));
                }
                for (ModLoadingIssue issue : incompatibilityIssues) {
                    issue = issue.withSeverity(reporting.getIssueSeverity());
                    LOGGER.atLevel(reporting.getLogLevel()).addMarker(LogMarkers.SCAN).log("Skipping jar. {}", (Object)FMLTranslations.translateIssueEnglish(issue));
                    this.addIssue(issue);
                }
            }
            ModDiscoverer.closeJarContents(jarContents);
            return Optional.empty();
        }

        @Override
        public boolean addModFile(IModFile mf) {
            Objects.requireNonNull(mf, "mf");
            if (!(mf instanceof ModFile)) {
                String detail = "Unexpected IModFile subclass: " + String.valueOf(mf.getClass());
                this.addIssue(ModLoadingIssue.error("fml.modloadingissue.technical_error", detail).withAffectedModFile(mf));
                return false;
            }
            ModFile modFile = (ModFile)mf;
            modFile.setDiscoveryAttributes(this.defaultAttributes.merge(mf.getDiscoveryAttributes()));
            ModFileDiscoveryAttributes discoveryAttributes = mf.getDiscoveryAttributes();
            LOGGER.info(LogMarkers.SCAN, "Found {} file \"{}\" {}", new Object[]{mf.getType().name().toLowerCase(Locale.ROOT), mf.getFileName(), discoveryAttributes});
            this.loadedFiles.add(modFile);
            ++this.successCount;
            return true;
        }

        @Override
        public void addIssue(ModLoadingIssue issue) {
            this.issues.add(issue);
            switch (issue.severity()) {
                case WARNING: {
                    ++this.warningCount;
                    break;
                }
                case ERROR: {
                    ++this.errorCount;
                }
            }
        }
    }

    public record Result(List<ModFile> modFiles, List<ModLoadingIssue> discoveryIssues) {
    }
}

