/*
 * Decompiled with CFR 0.152.
 */
package com.github.mizosoft.methanol.internal;

import com.github.mizosoft.methanol.internal.Validate;
import com.github.mizosoft.methanol.internal.text.HttpCharMatchers;
import java.io.Closeable;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.time.Clock;
import java.time.Duration;
import java.time.ZoneOffset;
import java.util.Objects;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.checkerframework.checker.nullness.qual.Nullable;

public class Utils {
    private static final System.Logger logger = System.getLogger(Utils.class.getName());
    private static final Clock SYSTEM_MILLIS_UTC = Clock.tickMillis(ZoneOffset.UTC);

    private Utils() {
    }

    public static boolean isValidToken(String token) {
        return !token.isEmpty() && HttpCharMatchers.TOKEN_MATCHER.allMatch(token);
    }

    public static String normalizeToken(String token) {
        Validate.requireArgument(Utils.isValidToken(token), "illegal token: '%s'", token);
        return Utils.toAsciiLowerCase(token);
    }

    public static void validateToken(String token) {
        Objects.requireNonNull(token);
        Validate.requireArgument(Utils.isValidToken(token), "illegal token: '%s'", token);
    }

    public static void validateHeaderName(String name) {
        Objects.requireNonNull(name);
        Validate.requireArgument(Utils.isValidToken(name), "illegal header name: '%s'", name);
    }

    public static void validateHeaderValue(String value) {
        Objects.requireNonNull(value);
        Validate.requireArgument(HttpCharMatchers.FIELD_VALUE_MATCHER.allMatch(value), "illegal header value: '%s'", value);
    }

    public static void validateHeader(String name, String value) {
        Utils.validateHeaderName(name);
        Utils.validateHeaderValue(value);
    }

    public static void requirePositiveDuration(Duration duration) {
        Objects.requireNonNull(duration);
        Validate.requireArgument(!duration.isNegative() && !duration.isZero(), "non-positive duration: %s", duration);
    }

    public static void requireNonNegativeDuration(Duration duration) {
        Objects.requireNonNull(duration);
        Validate.requireArgument(!duration.isNegative(), "negative duration: %s", duration);
    }

    public static int copyRemaining(ByteBuffer src, ByteBuffer dst) {
        int toCopy = Math.min(src.remaining(), dst.remaining());
        int srcLimit = src.limit();
        src.limit(src.position() + toCopy);
        dst.put(src);
        src.limit(srcLimit);
        return toCopy;
    }

    public static ByteBuffer copy(ByteBuffer buffer) {
        return ByteBuffer.allocate(buffer.remaining()).put(buffer).flip();
    }

    public static ByteBuffer copy(ByteBuffer source2, ByteBuffer target) {
        return target.remaining() >= source2.remaining() ? target.put(source2) : Utils.copy(source2);
    }

    public static Clock systemMillisUtc() {
        return SYSTEM_MILLIS_UTC;
    }

    private static RuntimeException rethrowAsyncIOFailure(Throwable cause, boolean wrapInterruptedException) throws IOException, InterruptedException {
        Throwable rethrownCause = Utils.tryGetRethrownCause(cause);
        if (rethrownCause instanceof RuntimeException) {
            throw (RuntimeException)rethrownCause;
        }
        if (rethrownCause instanceof Error) {
            throw (Error)rethrownCause;
        }
        if (rethrownCause instanceof IOException) {
            throw (IOException)rethrownCause;
        }
        if (rethrownCause instanceof InterruptedException) {
            if (wrapInterruptedException) {
                InterruptedIOException interruptedIO = new InterruptedIOException(cause.getMessage());
                interruptedIO.initCause(cause);
                throw interruptedIO;
            }
            throw (InterruptedException)rethrownCause;
        }
        throw new IOException(cause.getMessage(), cause);
    }

    private static @Nullable Throwable tryGetRethrownCause(Throwable cause) {
        String message = cause.getMessage();
        Class<?> causeType = cause.getClass();
        try {
            for (Constructor<?> constructor : causeType.getConstructors()) {
                Class<?>[] paramTypes = constructor.getParameterTypes();
                int len = paramTypes.length;
                if (len == 2 && paramTypes[0] == String.class && paramTypes[1] == Throwable.class) {
                    return (Throwable)constructor.newInstance(message, cause);
                }
                if (len == 1 && paramTypes[0] == String.class) {
                    return ((Throwable)constructor.newInstance(message)).initCause(cause);
                }
                if (len == 1 && paramTypes[0] == Throwable.class) {
                    return (Throwable)constructor.newInstance(cause);
                }
                if (len != 0) continue;
                return ((Throwable)constructor.newInstance(new Object[0])).initCause(cause);
            }
        }
        catch (ReflectiveOperationException reflectiveOperationException) {
            // empty catch block
        }
        return null;
    }

    public static Throwable getDeepCompletionCause(Throwable t2) {
        Throwable deeperCause;
        Throwable cause = t2;
        while ((cause instanceof CompletionException || cause instanceof ExecutionException) && (deeperCause = cause.getCause()) != null) {
            cause = deeperCause;
        }
        return cause;
    }

    public static <T> T block(Future<T> future) throws IOException, InterruptedException {
        try {
            return future.get();
        }
        catch (ExecutionException failed) {
            throw Utils.rethrowAsyncIOFailure(Utils.getDeepCompletionCause(failed), false);
        }
    }

    public static <T> T blockOnIO(Future<T> future) throws IOException {
        try {
            return future.get();
        }
        catch (ExecutionException failed) {
            try {
                throw Utils.rethrowAsyncIOFailure(Utils.getDeepCompletionCause(failed), true);
            }
            catch (InterruptedException unexpected) {
                throw new AssertionError((Object)unexpected);
            }
        }
        catch (InterruptedException interrupted) {
            throw new InterruptedIOException();
        }
    }

    public static void closeQuietly(@Nullable Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException ioe) {
                logger.log(System.Logger.Level.WARNING, "exception while closing: " + closeable, (Throwable)ioe);
            }
        }
    }

    public static String escapeAndQuoteValueIfNeeded(String value) {
        return Utils.isValidToken(value) ? value : Utils.escapeAndQuote(value);
    }

    private static String escapeAndQuote(String value) {
        StringBuilder escaped = new StringBuilder();
        CharBuffer buffer = CharBuffer.wrap(value);
        escaped.append('\"');
        while (buffer.hasRemaining()) {
            char c = buffer.get();
            if (c == '\"' || c == '\\') {
                escaped.append('\\');
            }
            escaped.append(c);
        }
        escaped.append('\"');
        return escaped.toString();
    }

    private static String toAsciiLowerCase(CharSequence value) {
        StringBuilder lower = new StringBuilder(value.length());
        for (int i = 0; i < value.length(); ++i) {
            lower.append(Character.toLowerCase(value.charAt(i)));
        }
        return lower.toString();
    }
}

