package net.minecraftforge.srg2source.rangeapplier;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import net.minecraftforge.srg2source.rangeapplier.RangeMap.RangeEntry;
import net.minecraftforge.srg2source.util.Util;
import net.minecraftforge.srg2source.util.io.ConfLogger;
import net.minecraftforge.srg2source.util.io.FolderSupplier;
import net.minecraftforge.srg2source.util.io.InputSupplier;
import net.minecraftforge.srg2source.util.io.OutputSupplier;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.io.ByteStreams;

public class RangeApplier extends ConfLogger<RangeApplier>
{
    @SuppressWarnings({ "unchecked", "resource" })
    public static void main(String[] args) throws IOException
    {
        // configure parser
        OptionParser parser = new OptionParser();
        {
            parser.acceptsAll(Arrays.asList("h", "help")).isForHelp();
            parser.accepts("srcRoot", "Source root directory to rename").withRequiredArg().ofType(File.class).isRequired(); // var=srcRoot
            parser.accepts("srcRangeMap", "Source range map generated by srg2source").withRequiredArg().ofType(File.class).isRequired(); // var=srcRangeMap
            parser.accepts("srgFiles", "Symbol map file(s), separated by ' '").withRequiredArg().ofType(File.class).isRequired(); // var=srgFiles
            parser.accepts("git", "Command to invoke git"); // var=git default="git"
            parser.accepts("lvRangeMap", "Original source range map generated by srg2source, for renaming local variables"); // var=lvRangeMap)  CSV instead?
            parser.accepts("mcpConfDir", "MCP configuration directory, for renaming parameters").withRequiredArg().ofType(File.class); // var=mcpConfDir
            parser.accepts("excFiles", "Parameter map file(s), separated by ' '").withRequiredArg().ofType(File.class); // var=excFiles
            //parser.accepts("no-dumpRenameMap", "Disable dumping symbol rename map before renaming"); // var="dumpRenameMap", default=True
            parser.accepts("dumpRangeMap", "Enable dumping the ordered range map and quit"); // var=dumpRangeMap, default=False
            parser.accepts("outDir", "The output folder for editted classes.").withRequiredArg().ofType(File.class); // default null
        }

        OptionSet options = parser.parse(args);

        if (options.has("help"))
        {
            parser.printHelpOn(System.out);
            System.exit(0);
        }

        //boolean dumpRenameMap = !options.has("no-dumpRenameMap");
        boolean dumpRangeMap = options.has("dumpRangeMap");

        // read range map, spit, and return

        if (dumpRangeMap)
        {
            RangeMap ranges = new RangeMap().read((File) options.valueOf("srcRangeMap"));
            for (String key : ranges.keySet())
            {
                for (RangeEntry info : ranges.get(key))
                {
                    System.out.println(info);
                }
            }
            return;
        }

        // setup RangeApplier.
        RangeApplier app = new RangeApplier().readSrg((List<File>) options.valuesOf("srgFiles"));

        // read parameter remaps
        if (options.has("mcpConfDir") && options.hasArgument("mcpConfDir"))
        {
            File conf = (File) options.valueOf("mcpConfDir");
            File primary = new File(conf, "joined.exc");
            List<File> secondaries = null;

            if (!primary.exists())
                primary = new File(conf, "packaged.exc");

            if (options.has("excFiles") && options.hasArgument("excFiles"))
                secondaries = (List<File>) options.valuesOf("excFiles");

            app.readParamMap(primary, secondaries);
        }

        // read local varaible map
        if (options.has("lvRangeMap") && options.hasArgument("lvRangeMap"))
        {
            app.readLvRangeMap((File) options.valueOf("lvRangeMap"));
        }

        //options.valuesOf("srgFiles");
        FolderSupplier srcRoot = new FolderSupplier((File) options.valueOf("srcRoot"));
        // output supplier..
        OutputSupplier outDir = (OutputSupplier) srcRoot;
        if (options.has("outDir"))
            outDir = new FolderSupplier((File) options.valueOf("outDir"));
        
        app.remapSources(srcRoot, outDir, (File) options.valueOf("srcRangeMap"), false);
        
        srcRoot.close();
        outDir.close();

        System.out.println("FINISHED!");
    }

    private SrgContainer    srg = new SrgContainer();
    private final RenameMap map = new RenameMap();

    // SRG stuff
    public RangeApplier readSrg(File srg)
    {
        this.srg.readSrg(srg);
        map.readSrg(this.srg);
        return this;
    }

    public RangeApplier readSrg(List<File> srgs)
    {
        srg.readSrgs(srgs);
        map.readSrg(this.srg);
        return this;
    }

    public RangeApplier readSrg(SrgContainer srgCont)
    {
        this.srg = srgCont;
        map.readSrg(this.srg);
        return this;
    }

    // excptors
    public RangeApplier readParamMap(File primaryExceptor, List<File> secondaryExceptors)
    {
        ExceptorFile prim = new ExceptorFile().read(primaryExceptor);
        ExceptorFile secondary = secondaryExceptors == null || secondaryExceptors.isEmpty() ? null : new ExceptorFile().read(secondaryExceptors);
        map.readParamMap(srg, prim, secondary);

        return this;
    }

    // LV map
    public RangeApplier readLvRangeMap(File lvRangeMap)
    {
        try
        {
            map.readLocalVariableMap(new LocalVarFile().read(lvRangeMap), srg);
        }
        catch (IOException e)
        {
            Throwables.propagate(e);
        }

        return this;
    }
    
    /**
     * Outputs the contents of the rename map.
     * Spits everything to the outLogger.
     */
    public void dumpRenameMap()
    {
        for (Entry<String, String> e : map.maps.entrySet())
        {
            log("RENAME MAP: " + e.getKey() + " -> " + e.getValue());
        }
    }

    /**
     * Actually remaps stuff.
     * @param inSupp
     * @param outSupp
     * @param rangeMap
     * @param annotate Marks all renamed symbols with a comment and the old name.
     * @throws IOException
     */
    public void remapSources(InputSupplier inSupp, OutputSupplier outSupp, File rangeMap, boolean annotate) throws IOException
    {
        RangeMap range = new RangeMap().read(rangeMap);

        for (String filePath : range.keySet())
        {
            InputStream stream = inSupp.getInput(filePath);
            String data = new String(ByteStreams.toByteArray(stream), Charset.forName("UTF-8"));
            stream.close();

            if (data.contains("\r"))
            {
                // to ensure that the offsets are not off by 1.
                log("Warning: " + filePath + " has CRLF line endings; consider switching to LF");
                data = data.replace("\r", "");
            }

            // process
            List<String> out = processJavaSourceFile(filePath, data, range.get(filePath), annotate);
            filePath = out.get(0);
            data = out.get(1);

            // write.
            OutputStream outStream = outSupp.getOutput(filePath);
            outStream.write(data.getBytes(Charset.forName("UTF-8")));
            outStream.close();
        }
    }

    // ---------------------------------------------------------------------------------------------------
    // ---------------------------------------------------------------------------------------------------
    // ---------------------------------------------------------------------------------------------------
    // ---------------------------------------------------------------------------------------------------

    /**
     * Rename symbols in source code
     * @return
     * @throws IOException
     */
    private ImmutableList<String> processJavaSourceFile(String fileName, String data, Collection<RangeEntry> rangeList, boolean shouldAnnotate) throws IOException
    {
        StringBuilder outData = new StringBuilder();
        outData.append(data);

        Set<String> importsToAdd = new TreeSet<String>();
        int shift = 0;

        // Existing package/class name (with package, internal) derived from filename
        String oldTopLevelClassFullName = Util.getTopLevelClassForFilename(fileName);
        String oldTopLevelClassPackage = Util.splitPackageName(oldTopLevelClassFullName);
        String oldTopLevelClassName = Util.splitBaseName(oldTopLevelClassFullName);

        // New package/class name through mapping
        String newTopLevelClassPackage = Util.sourceName2Internal(map.maps.get("package " + oldTopLevelClassPackage));
        String newTopLevelClassName = map.maps.get("class " + oldTopLevelClassFullName);
        if (newTopLevelClassPackage != null && newTopLevelClassName == null)
            throw new RuntimeException("filename " + fileName + " found package " + oldTopLevelClassPackage + "->" + newTopLevelClassPackage + " but no class map for " + newTopLevelClassName);
        if (newTopLevelClassPackage == null && newTopLevelClassName != null)
            throw new RuntimeException("filename " + fileName + " found class map " + oldTopLevelClassName + "->" + newTopLevelClassName + " but no package map for " + oldTopLevelClassPackage);

        // start,end,expectedOldText,key
        for (RangeEntry info : rangeList)
        {
            int end = info.end;
            String expectedOldText = info.expectedOldText;
            if (map.maps.containsKey(info.key) && map.maps.get(info.key).isEmpty()) // has an empty key.
            {
                // Replacing a symbol with no text = removing a symbol
                if (!info.key.startsWith("package "))
                    throw new RuntimeException("unable to remove non-package symbol " + info.key);

                // Remove that pesky extra period after qualified package names
                end++;
                expectedOldText += ".";
            }

            String oldName = outData.substring(info.start + shift, end + shift);

            if (!oldName.equals(expectedOldText))
                throw new RuntimeException("Rename sanity check failed: expected '" + expectedOldText + "' at [" + info.start + "," + end + "] (shifted " + shift + " to [" + (shift + info.start) + "," + (shift + end) + "]) in " + fileName + ", but found '" + oldName + "'\nRegenerate symbol map on latest sources or start with fresh source and try again");

            String newName = getNewName(info.key, oldName, map.maps, shouldAnnotate);
            if (newName == null)
            {
                if (info.key.split(" ")[1].contains("net/minecraft"))
                    log("No rename for " + info.key);
                continue;
            }

            log("Rename " + info.key + "[" + (info.start + shift) + "," + (end + shift) + "]" + "::" + oldName + "->" + newName);

            if (map.imports.containsKey(info.key))
            {
                // This rename requires adding an import, if it crosses packages
                String importPackage = Util.splitPackageName(Util.sourceName2Internal(map.imports.get(info.key)));
                if (!importPackage.equals(newTopLevelClassPackage))
                    importsToAdd.add(map.imports.get(info.key));
            }
            // Rename algorithm: 
            // 1. textually replace text at specified range with new text
            // 2. shift future ranges by difference in text length
            //data = data.substring(0, info.start + shift) + newName + data.substring(end + shift);
            outData.replace(info.start + shift, end + shift, newName);
            shift += (newName.length() - oldName.length());
        }

        // Lastly, update imports - this == separate from symbol range manipulation above
        String outString = updateImports(outData, importsToAdd, map.imports);

        // rename?
        if (newTopLevelClassPackage != null) // rename if package changed
        {
            String newFileName = (newTopLevelClassPackage + "/" + newTopLevelClassName + ".java").replace('\\', '/');

            log("Rename file " + fileName + " -> " + newFileName);

            fileName = newFileName;
        }

        return ImmutableList.of(fileName, outString);
    }

    /**
     * Add new import statements to source
     */
    private String updateImports(StringBuilder data, Set<String> newImports, Map<String, String> importMap)
    {
        //String[] lines = data.split("\n");

        int lastIndex = 0;
        int nextIndex = data.indexOf("\n");
        // Parse the existing imports and find out where to add ours
        // This doesn't use Psi.. but the syntax is easy enough to parse here
        boolean addedNewImports = false;
        boolean sawImports = false;

        String line;
        while (nextIndex > -1)
        {
            line = data.substring(lastIndex, nextIndex);

            while (line.startsWith("\n"))
            {
                lastIndex++;
                line = data.substring(lastIndex, nextIndex);
            }

            if (line.startsWith("import "))
            {
                sawImports = true;
                
                // remove stuff thats already added by a wildcard
                if (line.indexOf('*') > 0)
                {
                    LinkedList<String> remove = new LinkedList<String>();
                    String starter = line.replace("import ", "").replace(".*;", "");
                    for (String imp : newImports)
                    {
                        String impStart = imp.substring(0, imp.lastIndexOf('.'));
                        if (impStart.equals(starter))
                            remove.add(imp);
                    }
                    newImports.removeAll(remove);
                }

                if (line.startsWith("import net.minecraft."))
                {
                    // If no import map, *remove* NMS imports (OBC rewritten with fully-qualified names)
                    if (importMap.isEmpty())
                    {
                        // next line.
                        lastIndex = nextIndex + 1; // +1 to skip the \n at the end of the line there
                        nextIndex = data.indexOf("\n", lastIndex + 1); // another +1 because otherwise it would just return lastIndex
                        continue;
                    }

                    // Rewrite NMS imports
                    String oldClass = line.replace("import ", "").replace(";", "");
                    log("Import: " + oldClass);

                    String newClass = oldClass;
                    if (oldClass.equals("net.minecraft.server.*"))
                    {
                        // wildcard NMS imports (CraftWorld, CraftEntity, CraftPlayer).. bad idea
                        
                        // next line.  Duplicated from the bottom of the loop.
                        lastIndex = nextIndex + 1; // +1 to skip the \n at the end of the line there
                        nextIndex = data.indexOf("\n", lastIndex + 1); // another +1 because otherwise it would just return lastIndex
                        continue;
                    }
                    else if (importMap.containsKey("class " + Util.sourceName2Internal(oldClass)))
                        newClass = importMap.get("class " + Util.sourceName2Internal(oldClass));

                    if (newImports.contains(newClass))  // if not already added & its changed
                    {
                        if (oldClass.equals(newClass))
                        {
                            newImports.remove(newClass);
                        }
                        else
                        {
                            // otherwise remove from the file... it will be added again later.
                            data.delete(lastIndex, nextIndex + 1); // the newLine too
                            nextIndex = data.indexOf("\n", lastIndex); // get from here to the end of the line.
                            continue;
                        }
                    }
                    else
                    {
                        int change = "import ".length();
                        data.replace(lastIndex, nextIndex, "import ");
                        data.insert(lastIndex + change, newClass);
                        change += newClass.length();
                        data.insert(lastIndex + change, ";");
                        nextIndex = lastIndex + change + 1; // +1 for the semicolon
                    }
                }
            }
            else if (sawImports && !addedNewImports)
            {
                // Add our new imports right after the last import
                log("Adding " + newImports.size() + " imports");

                CharSequence sub = data.subSequence(lastIndex, data.length()); // grab the rest of the string.
                data.setLength(lastIndex); // cut off the build there

                for (String imp : newImports)
                    data.append("import ").append(imp).append(";\n");

                int change = data.length() - lastIndex; // get changed size
                lastIndex = data.length(); // reset the end to the actual end..
                nextIndex += change; // shift nextIndex accordingly..

                data.append(sub); // add on the rest if the string again

                addedNewImports = true;
            }

            // next line.
            lastIndex = nextIndex + 1; // +1 to skip the \n at the end of the line there
            nextIndex = data.indexOf("\n", lastIndex + 1); // another +1 because otherwise it would just return lastIndex
        }

        // got through the whole file without seeing or adding any imports???
        if (!addedNewImports)
        {
            // insert imports after the second line.
            int index = data.indexOf("\n") + 1;
            index = data.indexOf("\n", index) + 1; // search again from the second point, for 2 lines. +1 for after the \n

            CharSequence sub = data.subSequence(index, data.length()); // grab the rest of the string.
            data.setLength(index); // cut off the build there

            for (String imp : newImports)
                data.append("import ").append(imp).append(";\n");

            data.append(sub); // add on the rest if the string again
        }

        String newData = data.toString();

        // Warning: ugly hack ahead
        // The symbol range map extractor is supposed to emit package reference ranges, which we can 
        // update with the correct new package names. However, it has a bug where the package ranges
        // are not always emitted on fully-qualified names. For example: (net.minecraft.server.X)Y - a
        // cast - will fail to recognize the net.minecraft.server package, so it won't be processed by us.
        // This leads to some qualified names in the original source to becoming "overqualified", that is,
        // net.minecraft.server.net.minecraft.X; the NMS class is replaced with its fully-qualified name
        // (in non-NMS source, where we want it to always be fully-qualified): original package name isn't replaced.
        // Occurs in OBC source which uses fully-qualified NMS names already, and NMS source which (unnecessarily)
        // uses fully-qualified NMS names, too. Attempted to fix this problem for longer than I should.. 
        // maybe someone smarter can figure it out -- but until then, in the interest of expediency, I present 
        // this ugly workaround, replacing the overqualified names after-the-fact.
        // Fortunately, this pattern is easy enough to reliably detect and replace textually!
        newData = newData.replace("net.minecraft.server.net.minecraft", "net.minecraft");  // OBC overqualified symbols
        newData = newData.replace("net.minecraft.server.Block", "Block"); // NMS overqualified symbols
        // ..and qualified inner classes, only one.... last ugly hack, I promise :P
        newData = newData.replace("net.minecraft.block.BlockSapling/*was:BlockSapling*/.net.minecraft.block.BlockSapling.TreeGenerator", "net.minecraft.block.BlockSapling.TreeGenerator");
        newData = newData.replace("net.minecraft.block.BlockSapling.net.minecraft.block.BlockSapling.TreeGenerator", "net.minecraft.block.BlockSapling.TreeGenerator");

        return newData;
    }

    private String getNewName(String key, String oldName, Map<String, String> renameMap, boolean shouldAnnotate)
    {
        String newName;
        if (!renameMap.containsKey(key))
        {
            String constructorClassName = getConstructor(key);
            if (constructorClassName != null)
            {
                // Constructors are not in the method map (from .srg, and can't be derived
                // exclusively from the class map since we don't know all the parameters).. so we
                // have to synthesize a rename from the class map here. Ugh..but, it works.
                log("FOUND CONSTR " + key + " " + constructorClassName);
                if (renameMap.containsKey("class " + constructorClassName))
                    // Rename constructor to new class name
                    newName = Util.splitBaseName(renameMap.get("class " + constructorClassName));
                else
                    return null;
            }
            else
                // Not renaming this
                return null;
        }
        else
            newName = renameMap.get(key);
        
        newName = Util.splitBaseName(newName, Util.countChar(oldName, '.'));

        if (shouldAnnotate)
            newName += "/* was " + oldName + "*/";

        return newName;
    }

    /**
     * Check whether a unique identifier method key is a constructor, if so return full class name for remapping, else null
     */
    private String getConstructor(String key)
    {
        String[] tokens = key.split(" ", 3);  // TODO: switch to non-conflicting separator..types can have spaces :(
        if (!tokens[0].equals("method"))
            return null;
        log(Arrays.toString(tokens));
        //kind, fullMethodName, methodSig = tokens
        if (tokens[2].charAt(tokens[2].length() - 1) != 'V') // constructors marked with 'V' return type signature in ApplySrg2Source and MCP
            return null;
        String fullClassName = Util.splitPackageName(tokens[1]);
        String methodName = Util.splitBaseName(tokens[1]);

        String className = Util.splitBaseName(fullClassName);

        if (className.equals(methodName)) // constructor has same name as class
            return fullClassName;
        else
            return null;
    }

}
