package cpw.mods.jarhandling;

import cpw.mods.jarhandling.impl.Jar;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.CodeSigner;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

public interface SecureJar {
    Path getPrimaryPath();

    Optional<URI> findFile(String name);

    Manifest getManifest();

    CodeSigner[] getManifestSigners();

    Status verifyPath(Path path);

    Status getFileStatus(String name);

    Attributes getTrustedManifestEntries(String name);

    boolean hasSecurityData();

    static SecureJar from(final Path... paths) {
        return from(Manifest::new, jar -> JarMetadata.from(jar, paths), paths);
    }

    static SecureJar from(BiPredicate<String, String> filter, final Path... paths) {
        return from(Manifest::new, jar->JarMetadata.from(jar, paths), filter, paths);
    }

    static SecureJar from(Supplier<Manifest> defaultManifest, Function<SecureJar, JarMetadata> metadataSupplier, final Path... paths) {
        return from(defaultManifest, metadataSupplier, (e1, e2)->true, paths);
    }

    static SecureJar from(Supplier<Manifest> defaultManifest, Function<SecureJar, JarMetadata> metadataSupplier, BiPredicate<String, String> filter, final Path... paths) {
        return new Jar(defaultManifest, metadataSupplier, filter, paths);
    }

    Set<String> getPackages();

    List<Provider> getProviders();

    String name();

    Path getPath(String first, String... rest);

    Path getRootPath();

    record Provider(String serviceName, List<String> providers) {
        public static Provider fromPath(final Path path, final BiPredicate<String, String> pkgFilter) {
            final var sname = path.getFileName().toString();
            try {
                var entries = Files.readAllLines(path).stream()
                        .map(String::trim)
                        .filter(l->l.length() > 0 && !l.startsWith("#"))
                        .filter(p->pkgFilter.test(p.replace('.','/'), ""))
                        .toList();
                return new Provider(sname, entries);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    enum Status {
        NONE, INVALID, UNVERIFIED, VERIFIED
    }
}
